 ; Ŀ
 ;   Malaya - see if a layer exists and has the right colour and linetype. 
 ;   If not then take steps...                                             
 ;   Copyright 1999 - 2006 by Rocket Software Ltd.                         
 ;   If a layer exists but has the wrong colour or linetype, fix it.       
 ;   If it doesn't exist, make it, unless there is no layer data,          
 ;   in which case make it with the default settings.                      
 ;   Once it exists, make it current.                                      
 ;   This avoids having to insert the whole set of layers to get one.      
 ;                                                                         
 ;   Also contains utilities:                                              
 ;   Coid - Make all layers named in an internal list.                     
 ;   Llout - extract layer data into a file in the format Malaya likes.    
 ;   Fixla - make all layers in a drawing correct (if Malaya has data).    
 ;   Xlay - set up xref layers properly.                                   
 ;                                                                         
 ;   Correct layer settings are contained in (prlist) which must be        
 ;   edited if new layers are added or existing ones changed.              
 ;   Also C:Coid and C:Xlay contain editable internal data lists.          
 ;                                                                         
 ;   Notes:                                                                
 ;                                                                         
 ;   Malaya First checks to see if the layer name it has been handed is a  
 ;   variable, if so it uses the name it contains, otherwise the string    
 ;   is taken to be a layer name.  In either case it is checked against    
 ;   an internal list.                                                     
 ;                                                                         
 ;   This being the case, the standard layer names can be placed directly  
 ;   in the menu file, and if different names are required they can be     
 ;   used merely by considering them to be variables and setting them to   
 ;   the new names. In other words, we can maintain client layer files in  
 ;   specific job subdirectories.  (Layer data must be present in the      
 ;   layer data list LaList in Malaya.)                                    
 ;                                                                         
 ; 

 ; Ŀ
 ;   Coid - utility - make all layers named in a list using the setting    
 ;   data from the list made by Prlist.                                    
 ;   This could be modified to read a data list from an external file      
 ;   and thus use one of several client-specific layer sets.               
 ;   Put the default file in c:\acad\support and the client one (if any)   
 ;   in the job-specific start directory.                                  
 ; 
 (DEFUN C:COID (/ lalist lanam laydat)
 ; Might want to update this list before using.
  (setq maclst (list "c-building" "busbar" "e-busbar" "bus_bar" "cable"
                     "e-cable" "cableug" "e-cableug" "center" "e-center"
                     "cond_cable" "contin"))
 ; Ŀ
 ;   Call prlist to make the layer data list.                              
 ; 
  (setq lalist (prlist))
 ; Ŀ
 ;   Now step through the desired layer list and fix or make each one.     
 ; 
  (while (setq lanam (car maclst))
         (setq maclst (cdr maclst))
 ; Ŀ
 ;   See if there a a data sublist in lalist for the layer in question.    
 ;   If so call malaya to make or fix it.                                  
 ; 
         (setq lanam (strcase lanam t))
         (if (setq laydat (assoc lanam lalist))
             (malaya laydat)
 ; Ŀ
 ;   Otherwise print an error message, unless the layer is xref dependent. 
 ; 
             (if (/= 16 (logand 16 (cdr (assoc 70 clay))))
                 (prompt (strcat "\n** Nonstandard layer: " lanam " **")))))
 (princ))
 ; Ŀ
 ;   Coid end.                                                             
 ; 

 ; Ŀ
 ;   Fixla - utility - make sure the layer setting are correct for all     
 ;   layers in the current drawing.                                        
 ;   Also lists all nonstandard layers.                                    
 ; 
 (DEFUN C:FIXLA (/ lalist rewind clay lanam laydat)
 ; Ŀ
 ;   Call prlist to make the layer data list.                              
 ; 
  (setq lalist (prlist))
  (setq rewind T)
 ; Ŀ
 ;   Now step through the layer data table and fix each one as required.   
 ; 
  (while (setq clay (tblnext "layer" rewind))
         (setq rewind ())
         (setq lanam (cdr (assoc 2 clay)))
 ; Ŀ
 ;   See if there a a data sublist in lalist for the layer in question.    
 ;   If so call malaya to make or fix it.                                  
 ; 
         (setq lanam (strcase lanam t))
         (if (setq laydat (assoc lanam lalist))
             (malaya laydat)
 ; Ŀ
 ;   Otherwise print an error message, unless the layer is xref dependent. 
 ; 
             (if (/= 16 (logand 16 (cdr (assoc 70 clay))))
                 (prompt (strcat "\n** Nonstandard layer: " lanam " **")))))
 (princ))
 ; Ŀ
 ;   Fixla end.                                                            
 ; 

 ; Ŀ
 ;   Llout - utility - write the name, colour, and linetype of each layer  
 ;   to a file in the format that Malaya likes.                            
 ;   You can now copy layer settings out of a file for use in malaya.      
 ;   Consider going with an external layer list...maybe not, since one     
 ;   loses the advantage of an interpreted language.  Also would be        
 ;   reading a separate file to make each layer.                           
 ;   Maybe a separate subroutine at the start of the file to make Lalist.  
 ;   The sublist format is:  '(Layername colournumber linetype)            
 ; 
 (DEFUN C:LLOUT (/ rewind fnam clay clam cltype ccolo)
  (setq fnam (open "layer.log" "w"))
  (setq rewind T)
  (while (setq clay (tblnext "layer" rewind))
         (setq rewind ())
         (setq clam (cdr (assoc 2 clay)))
         (setq cltype (cdr (assoc 6 clay)))
         (setq ccolo (cdr (assoc 62 clay)))
         (print (list clam ccolo cltype) fnam))
  (close fnam)
 (princ))
 ; Ŀ
 ;   Llout end.                                                            
 ; 

 ; Ŀ
 ;   Xlay (utility): set up xref layers.                                   
 ; 
 (DEFUN C:XLAY (/ xlist clay rewind isxlay lanam clay)
 ; Ŀ
 ;   Make the xlayer data list.                                            
 ; 
 ; Might want to update this before using.
  (setq xlist '(("red"          8   "Thaw")
                ("dimension"    3   "Freeze")
                ("centre"       1   "Freeze")
                ("center"       1   "Freeze")
                ("text2"        8   "Freeze") 
                ("e-background" 5   "Freeze")
                ("bom"          2   "Freeze")))
 ; Ŀ
 ;   Now step through the layer data table and fix each one if required.   
 ; 
  (setq rewind T)
  (while (setq clay (tblnext "layer" rewind))
         (setq rewind ())
         (setq lanam (cdr (assoc 2 clay)))
         (setq lanam (strcase lanam t))
         (setq isxlay (if (= 16 (logand 16 (cdr (assoc 70 clay)))) T ()))
         (setq namend (cadr (splat "|" lanam)))
 ; Ŀ
 ;   If there a data sublist in xlist for the layer in question, and it    
 ;   is an xref dependent layer:                                           
 ;   If it is to be thawed, set the colour, if it is to be frozen do so.   
 ; 
         (if (and isxlay (setq laydat (assoc namend xlist)))
             (if (= (nth 2 laydat) "Thaw")
                 (command "layer" "colour" (cadr laydat) lanam "")
                 (command "layer" "freeze" lanam ""))))
 (princ))
 ; Ŀ
 ;   Xlay end.                                                             
 ; 

 ; Ŀ
 ;   Subroutine Lava - see if a layer exists and has the right colour      
 ;   and linetype.                                                         
 ;   If it exists but has the wrong colour or linetype, fix it.            
 ;   If it doesn't exist then make it.                                     
 ;                                                                         
 ;   Takes one argument, Laydat, a list: (layer_name colour linetype)      
 ;   Returns nothing.                                                      
 ; 
 (DEFUN LAVA (laydat / lanam coll lintyp ladata)
 ; Ŀ
 ;   Extract the correct data from the list.                               
 ; 
  (setq lanam (car laydat))
  (setq coll (nth 1 laydat))
  (setq lintyp (nth 2 laydat))
 ; Ŀ
 ;   Check the linetype, load it if it isn't already present.              
 ;   (Assumes that the linetypes are in the file Acad.lin.)                
 ; 
  (if (null (tblsearch "ltype" lintyp))
      (command "linetype" "load" lintyp "acad" ""))
 ; Ŀ
 ;   Get the layer data from the layer table.                              
 ; 
  (setq ladata (tblsearch "layer" lanam))
 ; Ŀ
 ;   If the layer doesn't exist of the properties are wrong, make/fix it.  
 ; 
  (cond ((null ladata)
         (command "layer" "new" lanam
                          "colour" coll lanam
                          "ltype" lintyp lanam ""))
        ((or (/= (cdr (assoc 6 ladata)) (strcase lintyp))
             (/= (cdr (assoc 62 ladata)) coll))
         (command "layer" "set" lanam
                          "colour" coll lanam
                          "ltype" lintyp lanam ""))))
 ; Ŀ
 ;   Subroutine Lava end.                                                  
 ; 

 ; Ŀ
 ;   Subroutine Prlist: make the layer names and desired properties list.  
 ;   Takes no arguments, returns a list.                                   
 ;                                                                         
 ;   The sublist format is:  '(Layername colournumber linetype)            
 ;   Notes:  1. Colour names won't work, you must use numbers.             
 ;           2. For those like myself who aren't always awake, you can't   
 ;              set the colour of a layer to "Bylayer".                    
 ;           3. Less obviously, you can't use "Byblock" (0) either.        
 ;                                                                         
 ;   Colour number equivalents:                                            
 ;                                                                         
 ;      Red        1                                                       
 ;      Yellow     2                                                       
 ;      Green      3                                                       
 ;      Cyan       4           Actual colour rendition may vary.           
 ;      Blue       5           Please consult your retailer.               
 ;      Magenta    6                                                       
 ;      White      7                                                       
 ;      Dark gray  8                                                       
 ;                                                                         
 ; 
 (DEFUN PRLIST ()
  (list '("0"                   7    "continuous")
 ; Ŀ
 ;   Some P&ID Layers.                                                     
 ; 
        '("flogen"              7    "continuous")
        '("flomain"             4    "continuous")
        '("flosec"              1    "continuous")
        '("floskidb"            2    "phantom")
 ; Ŀ
 ;   Leading 4s - I'm sure this made sense to someone.                     
 ; 
        '("4-block-1"           2    "continuous")
        '("4-block-2"           1    "continuous")
        '("4-block-3"           3    "continuous")
        '("4-dim-1"             2    "continuous")
        '("4-equip-0"           7    "continuous")  
        '("4-equip-1"           2    "continuous")
        '("4-equip-2"           1    "continuous")
        '("4-gnd-3"             3    "c13")   
        '("4-gndsym-1"          2    "continuous")         
        '("4-hatch-0"           7    "continuous")
        '("4-hatch-4"           4    "continuous")
        '("4-line-1"            2    "continuous")
        '("4-line-3"            3    "ph1")  
        '("4-text-1"            2    "continuous")
        '("4-text-2"            1    "continuous")
        '("4-text-3"            3    "continuous")
        '("4-tray-2"            1    "continuous")
        '("4-wire-0"            7    "da4")
        '("4-wire-1"            2    "continuous")
        '("4-wire-3"            3    "continuous")
 ; Ŀ
 ;   Leading Es with a dash - typically someone doing work for Encana.     
 ; 
        '("e-background"        5    "continuous")
        '("e-bom"               2    "continuous")
        '("e-busbar"            15   "continuous")   ; duplicate
        '("e-cableug"           15   "dashed")
        '("e-center"            3    "center")       ; duplicate
        '("e-dimension"         3    "continuous")
        '("e-eequip"            14   "continuous")   ; duplicate
        '("e-equipment"         2    "continuous")
        '("e-existing"          6    "continuous")
        '("e-field"             10   "dashed")
        '("e-frame"             13   "continuous")
        '("e-ground"            15   "center")       ; duplicate
        '("e-hidden"            6    "hidden")       ; duplicate
        '("e-instr"             10   "continuous")
        '("e-instrline"         6    "continuous")
        '("e-misc"              13   "continuous")   ; duplicate
        '("e-panel"             14   "continuous")
        '("e-shade"             1    "continuous")
        '("e-text1"             6    "continuous")   ; duplicate
        '("e-text2"             1    "continuous")   ; duplicate
        '("e-text3"             11   "continuous")   ; duplicate
        '("e-text4"             15   "continuous")
 ; Ŀ
 ;   Leading Es with a dash, but Rangeland settings - either these or      
 ;   the previous Es (duplicates are marked) will have to be commented     
 ;   out, or - since Malaya will use the first copy of a name which it     
 ;   finds - the desired ones can be moved higher up in the file.          
 ; 
        '("e-busbar"            4    "continuous")   ; duplicate
        '("e-cable"             3    "continuous")
        '("e-cable-ug"          3    "dashed")
        '("e-cable-tray"        165  "continuous")
        '("e-center"            1    "continuous")   ; duplicate
        '("e-eequip"            13   "continuous")   ; duplicate
        '("e-ground"            4    "phantom")      ; duplicate
        '("e-hidden"            1    "hidden")       ; duplicate
        '("e-misc"              5    "continuous")   ; duplicate
        '("e-text1"             2    "continuous")   ; duplicate
        '("e-text2"             3    "continuous")   ; duplicate
        '("e-text3"             4    "continuous")   ; duplicate
        '("e-wire-field"        3    "dashed")
        '("e-wire-panel"        3    "continuous")
 ; Ŀ
 ;   Leading Es with an underscore.                                        
 ; 
        '("e_busbar"            5    "continuous")
        '("e_cable"             4    "continuous")
        '("e_cable_control"     1    "continuous")
        '("e_cable_underground" 4    "hidden")
        '("e_cable_power"       4    "continuous")
        '("e_cln"               1    "center")
        '("e_communication"     2    "fenceline1")
        '("e_dim"               2    "continuous")
        '("e_areaclass"         3    "continuous")
        '("e_equipment"         3    "continuous")
        '("e_field_wiring"      122  "dashed")
        '("e_grounding"         4    "phantom")
        '("e_htc"               1    "continuous")
        '("e_heat_trace"        53   "continuous")
        '("e_key_boundary"      5    "border")
        '("e_mis"               7    "continuous")
        '("e_panel"             2    "continuous")
        '("e_panel_wiring"      42   "continuous")
        '("e_revision"          5    "continuous")
        '("e_symbols"           2    "continuous")
        '("e_txt"               2    "continuous")
 ; Ŀ
 ;   Other leading letters - obviously relics from an earlier time.        
 ; 
        '("a-revise"            11   "continuous")
        '("a-titleblock"        13   "continuous")
        '("c-building"          1    "continuous")
        '("t-topo"              3    "continuous")
 ; Ŀ
 ;   Layers named mostly after what they should contain.                   
 ; 
        '("areaclass"           3    "continuous")
        '("attrib"              7    "continuous")
        '("building"            1    "continuous")
        '("background"          8    "continuous") 
        '("base_equip"          253  "continuous")
        '("base_text"           41   "continuous")
        '("base_struct"         201  "continuous")
        '("base_bldg"           30   "continuous")
        '("bom"                 2    "continuous") 
        '("border1"             4    "continuous")
        '("border2"             6    "continuous")
        '("building"            1    "continuous") 
;       '("busbar"              4    "continuous") 
        '("busbar"              5    "continuous")  ; Tridyne
        '("bus_bar"             4    "continuous")
        '("cable"               4    "continuous") 
        '("cable_control"       1    "continuous")  ; Tridyne
        '("cable_power"         4    "continuous")  ; Tridyne
        '("cable_tray"          62   "continuous")  ; Tridyne
        '("cable_tray_hatch"    7    "continuous")  ; Tridyne
        '("cable_underground"   4    "hidden")      ; Tridyne
        '("cableug"             4    "dashed") 
        '("center"              3    "center") 
        '("communication"       2    "continuous")  ; Tridyne
        '("cond_cable"          7    "continuous")
        '("contin"              6    "continuous")
        '("defpoints"           7    "continuous") 
;       '("dimension"           3    "continuous")
        '("dimension"           1    "continuous")  ; Tridyne
        '("eequip"              7    "continuous") 
        '("elec_text"           7    "continuous")
        '("elec_equip"          4    "continuous")
        '("elec_cond"           1    "continuous")
        '("elec_instr"          201  "continuous")
        '("elec_exist"          201  "continuous")
;       '("equipment"           2    "continuous")
        '("equipment"           3    "continuous")  ; Tridyne
        '("existing"            6    "continuous") 
        '("fadeout"             8    "continuous")
;       '("field"               2    "dashed") 
        '("field"               62   "continuous")  ; Tridyne
        '("field_wiring"        122  "dashed")      ; Tridyne
        '("frame"               7    "continuous") 
        '("future"              6    "phantom2") 
        '("gnd_text"            21   "continuous")
        '("gnd_equip"           90   "continuous")
        '("gnd_tray"            183  "center")
        '("gnd_wire"            92   "center")
        '("ground"              4    "center") 
        '("grounding"           4    "phantom")     ; Tridyne
        '("ground_wire"         1    "center")
;       '("hatch"               6    "continuous")
        '("hatch"               7    "continuous")  ; Tridyne
        '("heat_trace"          1    "continuous")  ; Tridyne
        '("hidden"              6    "hidden")
        '("htrace"              5    "continuous")
        '("instr"               2    "continuous")
        '("instrline"           6    "continuous")
        '("instr_text"          140  "continuous")
        '("instr_equip"         6    "continuous")
        '("instr_dcond"         150  "continuous")
        '("instr_acond"         61   "continuous")
        '("instr_xcond"         191  "continuous")
        '("instr_exist"         233  "continuous")
        '("light_equip"         31   "continuous")
        '("light_cond"          112  "continuous")
        '("light_text"          221  "continuous")
        '("matchlines"          4    "phantom")     ; Tridyne
        '("misc"                1    "continuous")  ; Tridyne
;       '("misc"                7    "continuous")
;       '("panel"               7    "continuous") 
        '("panel"               2    "continuous")  ; Tridyne
        '("panel_wiring"        42   "continuous")  ; Tridyne
        '("pneu_text"           40   "continuous")
        '("pneu_equip"          131  "continuous")
        '("pneu_flow"           3    "continuous")
        '("pneu_tube"           200  "continuous")
        '("revise"              5    "continuous") 
        '("rotulo"              7    "continuous")
        '("shade"               255  "continuous")
        '("symbols"             2    "continuous")
;       '("text"                1    "continuous")
        '("text"                2    "continuous")  ; Tridyne
        '("text1"               1    "continuous")
        '("text2"               1    "continuous")
        '("text3"               5    "continuous")
        '("text4"               4    "continuous")
        '("title"               2    "continuous")
        '("titleblock"          7    "continuous")
        '("title_text"          2    "continuous")
        '("topo"                3    "continuous")
        '("tray"                2    "continuous")
        '("tray_text"           52   "continuous")
        '("tray_elec"           122  "continuous")
        '("tray_instr"          241  "continuous")
        '("ugnd_elec"           73   "hidden")
        '("ugnd_instr"          43   "hidden")
        '("ugnd_duct"           231  "continuous")
        '("ugnd_text"           254  "continuous")
        '("ugnd_struct"         181  "continuous")
        '("ugnd_equip"          132  "continuous")))
 ; Ŀ
 ;   Subroutine Prlist end.                                                
 ; 

 ; Ŀ
 ;   Subroutine Splat - divide a text string into a list of substrings.    
 ;   Arguments: Sepchr, the field separator character.                     
 ;              Linn, the text string.                                     
 ;   Returns a list of field values, removes leading and trailing spaces.  
 ;                                                                         
 ;   This could probably replace Split wherever it is called...            
 ; 
 (DEFUN SPLAT (sepchr linn / len pos name1 strlst)
  (while (/= (strlen linn) 0)
         (while (and (= (substr linn 1 1) " ")
                     (/= (strlen linn) 0))
                (setq linn (substr linn 2)))
         (while (= (substr linn (setq len (strlen linn))) " ")
                (setq linn (substr linn 1 (1- len))))
         (setq pos 1)
         (setq len (strlen linn))
         (while (and (/= (substr linn pos 1) sepchr)
                     (>= len pos))
                (setq pos (1+ pos)))
         (setq name1 (substr linn 1 (1- pos)))
         (while (= (substr name1 (setq len (strlen name1))) " ")
                (setq name1 (substr name1 1 (1- len))))
         (setq linn (substr linn (1+ pos)))
         (setq strlst (append strlst (list name1))))
  (if (null strlst) (setq strlst (list "")))
  strlst)
 ; Ŀ
 ;   Splat end.                                                            
 ; 

 ; Ŀ
 ;   Subroutine Malaya - the engine.                                       
 ;   Takes one argument, a layer name string.                              
 ; 
 (DEFUN MALAYA (lanam / regg exp lalist laydat coll lintyp ladata)
 ; Ŀ
 ;   Since this is intended to work with layer name variables as well as   
 ;   layer name strings eval the input so that the variable Lanam          
 ;   contains a string in either case.                                     
 ; 
  (if (= (type (setq lamp (eval (read lanam)))) 'STR)
      (setq lanam lamp))
 ; Ŀ
 ;   Save settings and reset.                                              
 ; 
  (setq regg (getvar "regenmode"))
  (setvar "regenmode" 0)
  (setq exp (getvar "expert"))
  (setvar "expert" 1)
 ; Ŀ
 ;   Call prlist to make the layer data list.                              
 ; 
  (setq lalist (prlist))
 ; Ŀ
 ;   See if there a a data sublist in lalist for the layer in question.    
 ;   If so call Lava to make or fix it.                                    
 ; 
  (setq lanam (strcase lanam t))
  (if (setq laydat (assoc lanam lalist))
      (lava laydat)
 ; Ŀ
 ;   Otherwise: print an error message, and if the layer doesn't exist     
 ;   make it with the default properties.                                  
 ; 
      (progn
           (prompt (strcat "** No data for layer " lanam
                           ": see Malaya.lsp **"))
           (if (null (tblsearch "layer" lanam))
               (command "layer" "make" lanam ""))))
 ; Ŀ
 ;   If the layer exists (and if we are here it should) make it current.   
 ; 
  (if (tblsearch "layer" lanam) (command "layer" "set" lanam ""))
 ; Ŀ
 ;   Reset everything.                                                     
 ; 
  (setvar "regenmode" regg)
  (setvar "expert" exp)
 (princ))
 ; Ŀ
 ;   Subroutine Malaya end.                                                
 ; 

 ; Ŀ
 ;   Malaya.                                                               
 ;   This is just a string input function, so that the program can be      
 ;   called from the command line or by another routine.                   
 ; 
 (DEFUN C:MALAYA (/ lanam)
  (setvar "cmdecho" 0)
 ; Ŀ
 ;   Ask for the name of the desired current layer.                        
 ;   This is designed to be used as part of a menu so there is no prompt.  
 ; 
  (setq lanam (getstring))
 ; Ŀ
 ;   Call Malaya to do the work.                                           
 ; 
  (malaya lanam)
 (princ))